﻿using Microsoft.Data.SqlClient;
using SoftPi.Tariscope.Common;
using SoftPI.Tariscope.WebAdministration.HotelSystems.Scripting.Interfaces;
using System;
using System.Collections.Generic;
using System.IO;

// Reference implementation of PMS UNI protocol. Use Serial port for communication with PMS
// Protocol described in HRS corporate standard. FIDELIO and OPERA PMS are supported

// Configurations possible:
// RecordFormat      variable describes format of call record
// RECORD_START -     string to send before call record
// RECORD_END -       string to send after call record
// ACK,NAK and ENQ - protocol defined chars 

public class UNIScriptRef : IScript
{
    private IScriptHost Host;

    private bool NeedFinish = false;
    private System.Threading.Thread ProcessingThread;


    const int RETRY_COUNT = 10;
    const int RETRY_WAIT = 10000; // 10 seconds
    const byte ACK = 6;
    const byte NAK = 15;
    const byte ENQ = 5;

    // Const READY_REQUEST as String = ENQ.ToString()       'Request from CAS
    // Const READY_ANSWER = ACK        'Positive answer from PMS
    // Const ERROR_ANSWER = NAK        'Negative answer from PMS

    private string RECORD_START = Convert.ToChar(2).ToString();        // Send this before call (STX)
    private string RECORD_END = Convert.ToChar(3).ToString();          // ETX

    const string RecordFormat = "P {2,6} {1:dd/MM/yy} {1:HH:mm} {5,16} {3:HH:mm:ss} {4:0000.00} N";

    public void Init(IScriptHost host)
    {
        this.Host = host;
        host.Close += OnClose;
        NeedFinish = false;
        ProcessingThread = new System.Threading.Thread(ThreadProcessing);
        ProcessingThread.Name = "Hotel-UNI script thread";
        ProcessingThread.Start();
    }
    private void OnClose(ref bool Cancel)
    {
        if (ProcessingThread != null)
        {
            NeedFinish = true;
            ProcessingThread.Join();
        }

        return;
    }
    public void Main(object Parameters)
    {
    }

    // 1.  b   Латинская прописная буква "b"
    // 2.  xxx номер записи по порядку начиная с 000 и шагом 1 до 999  далее с 000.
    // 3.  A_ACK_  константа.
    // 4.  DD/MM   дата телефонного разговора день/месяц
    // 5.  RRRRR   номер телефонного аппарата с которого звонили, выравненный влево если меньше 5 символов.
    // 6.  TT:TT   время начала звонка часы:минуты
    // 7.  DDDD    Проолжительность звонка в минутах ( например 0345= 345 мин )
    // 8.  MMMMMMM.MMM стоимость звонка в копейках или центах. Точка просто так. Выравнивается вправо ( Значение 1.234 = 12 руб 34 коп )
    // 9.  (17)P   Номер куда звонили. Выравнивается влево
    // 10. С   Тип звонка L-местный,F-международный, любая буква другая буква может использовать для третьего типа звонка ( например межгород, интернет и.т.д. ) и в моей программе я назначу тип звонка по умолчанию для всех нераспознанных типов (  L и  F )
    // 11. cr  Char(13) ---Возврат каретки
    // 12. lf  Char(10) --- Новая строка


    public void ThreadProcessing()
    {
        int RecNum = 0;

        // Possible answers after Call send
        List<byte[]> ListenFor = new List<byte[]>();
        ListenFor.Add(new byte[] { ACK });
        ListenFor.Add(new byte[] { NAK });

        // wait for response 10 seconds.        
        Host.DefaultTimeout = RETRY_WAIT;

        Host.AddEvent("ThreadProcessing", 6);

        try
        {
            while (true)
            {
                // get first item from list
                List<CallItemExport> Items = Host.GetItems(1);

                // if item exists try to send it
                if (Items.Count > 0)
                {
                    foreach (CallItemExport itm in Items)
                    {
                        try
                        {
                            Host.AddEvent(string.Format("Sending call: {0} - {1}", itm.DN, itm.Dialnumber), 6);

                            if (IsCallMadeByServiceAccount(itm.ID))
                            {
                                Host.AddEvent("Call is made by service account, skipping", 6);
                                Host.ItemRemove(1);
                                continue;
                            }
                            // Preparing PMS to receive call
                            Host.SendData(new byte[] { ENQ });
                            for (int i = 1; i <= RETRY_COUNT; i++)
                            {
                                if (NeedFinish)
                                    return;
                                if (!Host.WaitFor(new byte[] { ACK }))
                                {
                                    Host.AddEvent("No answer from PMS, retrying", 3);
                                    Host.Wait(RETRY_WAIT);
                                    if (i == RETRY_COUNT)
                                    {
                                        Host.AddEvent("No answer from PMS, taking a break", 2);
                                        Host.Wait(RETRY_WAIT * 10);
                                        continue;
                                    }
                                }
                                else
                                    break;// answer received from PMS, sending call
                            }

                            // seems to be PMS ready. sending call
                            string Message = string.Format(System.Globalization.CultureInfo.InvariantCulture, RecordFormat, RecNum, itm.Datetime, itm.DN, new TimeSpan(0, 0, itm.DurationSeconds).ToString(), itm.Cost, itm.Dialnumber);
                            byte BCCValue = CalculateBCC(System.Text.Encoding.Default.GetBytes(Message));

                            Host.Send(RECORD_START + Message + RECORD_END + BCCValue);

                            long SendResult = Host.WaitForMultiple(ListenFor);
                            if (SendResult == 0)
                            {
                                // Success
                                RecNum += 1;
                                if (RecNum > 999)
                                    RecNum = 0;

                                Host.ItemRemove(1);
                                Host.AddEvent("Received OK from PMS.", 6);
                            }
                            else if (SendResult == 1)
                            {
                                Host.AddEvent("Call rejected.", 2);
                                Host.Wait(RETRY_WAIT);
                            }
                            else
                                Host.AddEvent("Answer timeout, will try again.", 3);
                        }
                        catch (IOException ex)
                        {
                        }
                    }
                }
                if (NeedFinish)
                    break;
                Host.Wait(100);
            }
        }
        catch (System.Threading.ThreadAbortException)
        {
            Host.AddEvent("Export thread terminated", 6);
        }
        ProcessingThread = null;
        return;
    }

    public byte CalculateBCC(byte[] indata)
    {
        byte result = 0;
        foreach (byte bt in indata)
            result ^= bt;
        return result;
    }

    public bool IsCallMadeByServiceAccount(long id)
    {
        Host.AddEvent("Call id: " + id, 6);

        try
        {
            using (var cn = new SqlConnection(Host.Configuration.DatabaseConnectionString))
            {
                cn.Open();
                string cmdText = string.Format("SELECT A.AbonentType FROM viCalls AS C INNER JOIN Abonents AS A ON C.FromAbonentID=A.ID WHERE C.Id={0} AND C.FromAbonent IS NOT NULL", id);
                var cmd = new SqlCommand(cmdText, cn);
                using (var rs = cmd.ExecuteReader())
                    if (rs.Read())
                        return rs.GetInt32(0) == 2;
            }
        }
        catch (Exception ex)
        {
            Host.AddEvent("IsCallMadeByServiceAccount error:", 6);
            Host.AddEvent("Exception message: " + ex.Message, 6);
            return false;
        }
        return false;
    }
}